/*
 * Copyright (C) 2015 Jiada Wang <jiada_wang@mentor.com>
 * Copyright (C) 2015 Mentor Graphics
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation, and
 * that the name of the copyright holders not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no representations
 * about the suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <string.h>
#include <netinet/in.h>

#include "dgram_service.h"

int dgram_recv(struct sk_dgram *skd, void *ubuf, int ulen)
{
	struct iovec iov[1];
	struct msghdr msg;
	int ret;

	memset(&msg, 0, sizeof(msg));
	memset(iov, 0, sizeof(iov));

	if (skd->proto != AF_INET) {
		iov[0].iov_base = ubuf;
		iov[0].iov_len = ulen;
		msg.msg_iov = iov;
		msg.msg_iovlen = 1;

		ret = recvmsg(skd->sk, &msg, 0);
		if (ret <= 0) {
			ret = -errno;
			debug_print("receive failed with err %d\n", ret);
		}

		return ret;
	}

	if ((skd->received < skd->hlen) || ((skd->received >= skd->hlen) &&
			(skd->received < (skd->h.dglen + skd->hlen)))) {
		iov[0].iov_base = skd->buf + skd->received;
		iov[0].iov_len = (skd->len + skd->hlen) - skd->received;
		msg.msg_iov = iov;
		msg.msg_iovlen = 1;

		ret = recvmsg(skd->sk, &msg, 0);
		if (ret <= 0) {
			ret = -errno;
			debug_print("receive failed with err %d\n", ret);
			return ret;
		}

		skd->received += ret;
	}

	if (skd->received < skd->hlen)
		return -EAGAIN;

	/*complete header available*/
	memcpy((char *)&skd->h, skd->buf, skd->hlen);
	skd->h.dglen = ntohs(skd->h.dglen);

	if (skd->received < (skd->h.dglen+skd->hlen))
		return -EAGAIN;

	/*full dgram available*/
	if (ulen < skd->h.dglen) {
		fprintf(stderr, "recv: dgram exceeds bufsize (%d > %d)\n",
				skd->h.dglen, ulen);
		return -EMSGSIZE;
	}

	memcpy(ubuf, skd->buf + skd->hlen, skd->h.dglen);
	skd->received -= (skd->h.dglen + skd->hlen);
	ret = skd->h.dglen;
	if (skd->received > 0) {
		memmove(skd->buf, skd->buf + skd->hlen + skd->h.dglen,
				skd->received);
		if (skd->received >= skd->hlen) {
			memcpy((char *)&skd->h, skd->buf, skd->hlen);
			skd->h.dglen = ntohs(skd->h.dglen);
		}
	}

	debug_print("finished DGRAM of len %d\n", ret);
	return ret;
}

int dgram_send(struct sk_dgram *skd, void *ubuf, int ulen)
{
	struct msghdr msg;
	struct iovec iov[2];
	struct dgram_header_std h;
	int ret;

	memset(&msg, 0, sizeof(msg));
	memset(iov, 0, sizeof(iov));

	if (skd->proto != AF_INET) {
		h.dglen = htons(ulen);
		iov[0].iov_base = ubuf;
		iov[0].iov_len = ulen;
		msg.msg_iov = iov;
		msg.msg_iovlen = 1;

		ret = sendmsg(skd->sk, &msg, 0);
		if (ret < 0) {
			ret = -errno;
			fprintf(stderr, "could not send msg: %d\n", ret);
		}

		return ret;
	}

	if (ulen > skd->len) {
		fprintf(stderr, "send: dgram exceeds bufsize (%d > %d)\n",
			ulen, skd->len);
		return -EMSGSIZE;
	}

	h.dglen = htons(ulen);
	iov[0].iov_base = &h;
	iov[0].iov_len = sizeof(h);
	iov[1].iov_base = ubuf;
	iov[1].iov_len = ulen;
	msg.msg_iov = iov;
	msg.msg_iovlen = 2;

	ret = sendmsg(skd->sk, &msg, 0);
	if (ret < 0) {
		ret = -errno;
		fprintf(stderr, "could not send msg: %d\n", ret);
	}

	if (ret >= sizeof(h))
		ret -= sizeof(h);

	return ret;
}

struct sk_dgram *dgram_init(int sk, int dgram_max, int proto)
{
	struct sk_dgram *skd;

	if (dgram_max > DGRAM_MAX) {
		fprintf(stderr, "exceed maxsize (%d > %d)\n", dgram_max, (int)DGRAM_MAX);
		return NULL;
	}

	switch(proto) {
	case AF_INET:
	case AF_INC:
		break;
	default:
		fprintf(stderr, "not supported on AF %d\n", proto);
		return NULL;
	}

	skd = calloc(1, sizeof(struct sk_dgram));
	if (!skd) {
		fprintf(stderr, "failed to alloc memory\n");
		return NULL;
	}

	skd->proto = proto;
	skd->hlen = sizeof(struct dgram_header_std);
	skd->len = dgram_max;
	skd->buf = calloc(skd->len + skd->hlen, sizeof(char));
	if (!skd->buf) {
		fprintf(stderr, "failed to alloc rcv buffer\n");
		free(skd);
		return NULL;
	}

	skd->sk = sk;

	return skd;
}

void dgram_exit(struct sk_dgram *skd)
{
	free(skd->buf);
	free(skd);
}
